<?php

/**
 * Class HubCheckoutHelper
 */
class HubKeyGenerator
{
    /**
     * @throws SodiumException
     */
    public function generateCustomerEncryptionKey(string $email, string $password): ?string
    {
        // Check if Sodium extension is available
        if (!$this->isShopSupportEncryption()) {
            return null; // Return null string if Sodium is not available
        }
        // Use the same hashed password and salt every time
        $password = hash('sha256', trim($password));

        // Always the same for a given email
        $salt = $this->createSaltFromCustomerEmail($email);

        // Generate the key based on the consistent password and salt
        $key = sodium_crypto_pwhash(
            SODIUM_CRYPTO_SECRETBOX_KEYBYTES,
            $password,
            $salt,
            SODIUM_CRYPTO_PWHASH_OPSLIMIT_INTERACTIVE,
            SODIUM_CRYPTO_PWHASH_MEMLIMIT_INTERACTIVE
        );

       return sodium_bin2base64($key, SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING);
    }

    /**
     * Using customer email to generate salt
     *
     * @param string $email
     * @return string
     */
    private function createSaltFromCustomerEmail(string $email): string {
        $hash = hash('sha256', trim(mb_strtolower($email)), true);

        return substr($hash, 0, SODIUM_CRYPTO_PWHASH_SALTBYTES);
    }

    /**
     * Check is shop support Encryption
     * @return bool 
     */
    private function isShopSupportEncryption(): bool
    {
        return function_exists('sodium_crypto_pwhash')
            && function_exists('sodium_bin2base64')
            && defined('SODIUM_CRYPTO_SECRETBOX_KEYBYTES')
            && defined('SODIUM_CRYPTO_PWHASH_OPSLIMIT_INTERACTIVE')
            && defined('SODIUM_CRYPTO_PWHASH_MEMLIMIT_INTERACTIVE')
            && defined('SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING')
            && defined('SODIUM_CRYPTO_PWHASH_SALTBYTES');
    }
}
